/* ==================================================================
File        : copytex.hlsl
Author      : various
Date        : 2007
Format      : HLSL/Cg
Description : copying operators

================================================================== */

#include "default.hlsl"

//------------------------------------------------------------------------------------------------
// Texture samplers
//------------------------------------------------------------------------------------------------

#define diffuse_tex layer0_sampler

//@:-texformat glow_tex RGBA8
//@:-texformat stencilValues_tex RGBA8
//@:-texformat diffuse1_tex RGBA8
sampler2D glow_tex			: register(s1);
sampler2D stencilValues_tex	: register(s1);
sampler2D diffuse1_tex		: register(s2);
sampler2D depth_tex			: register(s5);

//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------

uniform float diffuse_tex_lod	: register(c128);
uniform float threshold			: register(c30); // copytexthreshold
uniform float2 depth_modifiers	: register(c175);
uniform float4 mips[10]			: register(c180);

static const float EDRAM_TILE_WIDTH = 80;

//_________________________________________________________________________________________________

float4 copytex_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	return tex2D(diffuse_tex, In.uvSet01.xy);
}

//_________________________________________________________________________________________________

void copytexcolordepth_px(	in VS_OUT_POS_UVSET01 In,
							out float4 outColor : COLOR,
							out float outDepth : DEPTH)
{
	outColor = tex2D(diffuse_tex, In.uvSet01.xy);
	outDepth = tex2D(depth_tex, In.uvSet01.xy).r;
}

//_________________________________________________________________________________________________

void copytexFastdepth_px(	in VS_OUT_POS_UVSET01 In,
							out float4 outDepthAsColor : COLOR)
{
	float2 texCoord = In.uvSet01.xy;

	// The Microsoft paper recommends using VPOS here. However, I found that using
	// a normal interpolant is actually much faster (0.1ms)
	float columnIndex = texCoord.x / (EDRAM_TILE_WIDTH / 1280);
	float halfColumn = frac(columnIndex);

	if (halfColumn>=0.5)
		texCoord.x -= (EDRAM_TILE_WIDTH / 1280) / 2;
	else
		texCoord.x += (EDRAM_TILE_WIDTH / 1280) / 2;

	float4 result = tex2D(depth_tex, texCoord);

	//outDepthAsColor = result.bgra;
	outDepthAsColor = float4(0, result.gra);	// Set stencil to 0
}

//_________________________________________________________________________________________________

void copytexcolorFastdepth_px(	in VS_OUT_POS_UVSET01 In,
								out float4 outColor : COLOR0,
								out float4 outDepthAsColor : COLOR1)
{
	float2 texCoord = In.uvSet01.xy;

	// The Microsoft paper recommends using VPOS here. However, I found that using
	// a normal interpolant is actually much faster (0.1ms)
	float columnIndex = texCoord.x / (EDRAM_TILE_WIDTH / 1280);
	float halfColumn = frac(columnIndex);
	
	float2 texCoordDepth = texCoord;
	if (halfColumn>=0.5)
		texCoordDepth.x -= (EDRAM_TILE_WIDTH / 1280) / 2;
	else
		texCoordDepth.x += (EDRAM_TILE_WIDTH / 1280) / 2;

	float4 result = tex2D(depth_tex, texCoordDepth);

	outColor = tex2D(diffuse_tex, texCoord);

	//outDepthAsColor = result.bgra;
	outDepthAsColor = float4(0, result.gra);	// Set stencil to 0
}

//_________________________________________________________________________________________________

void copytexdepth_px(	in VS_OUT_POS_UVSET01 In,
						out float4 outColor : COLOR,
						out float outDepth : DEPTH)
{
	outColor = 1.0; // Fxc generates an error if we dont write a color value
	outDepth = tex2D(depth_tex, In.uvSet01.xy).r;
}

//_________________________________________________________________________________________________

float4 copytexHDR_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float4 src = tex2D(diffuse_tex, In.uvSet01.xy);

	float intensity = dot( src.rgb, float3(0.33,0.33,0.33) );
	float bloomIntensity = pow( intensity, 6.0 );
	src.rgb = src.rgb * bloomIntensity / intensity;

	//float3 col = saturate( src.rgb*2.0 - 1.0 );
	//src.rgb = col.rgb * col.rgb * col.rgb;

	return src;
}
//_________________________________________________________________________________________________

float4 copytexHDRAlpha_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float4 src = tex2D(diffuse_tex, In.uvSet01.xy);
	src.rgb *= src.a;
	return src;
}

//_________________________________________________________________________________________________

float4 copytexmip2_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float4 UV1 = float4(In.uvSet01.xy, 0.0, mips[0].x);
	float4 UV2 = float4(In.uvSet01.xy, 0.0, mips[1].x);

	float4 result    = tex2Dlod(diffuse_tex, UV1) * mips[0].y;
	result			+= tex2Dlod(diffuse_tex, UV2) * mips[1].y;
	
	// Chromatic aberration
	result.r = pow( result.r, 1.25 );
	result.g = pow( result.g, 1.10 );
	result.b = result.b;

	return result;
}

//_________________________________________________________________________________________________

float4 copytexmip3_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float4 UV1 = float4(In.uvSet01.xy, 0.0, mips[0].x);
	float4 UV2 = float4(In.uvSet01.xy, 0.0, mips[1].x);
	float4 UV3 = float4(In.uvSet01.xy, 0.0, mips[2].x);

	float4 result    = tex2Dlod(diffuse_tex, UV1) * mips[0].y;
	result			+= tex2Dlod(diffuse_tex, UV2) * mips[1].y;
	result			+= tex2Dlod(diffuse_tex, UV3) * mips[2].y;
	
	// Chromatic aberration
	result.r = pow( result.r, 1.25 );
	result.g = pow( result.g, 1.10 );
	result.b = result.b;

	return result;
}

//_________________________________________________________________________________________________

float4 copytexmip4_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float4 UV1 = float4(In.uvSet01.xy, 0.0, mips[0].x);
	float4 UV2 = float4(In.uvSet01.xy, 0.0, mips[1].x);
	float4 UV3 = float4(In.uvSet01.xy, 0.0, mips[2].x);
	float4 UV4 = float4(In.uvSet01.xy, 0.0, mips[3].x);

	float4 result    = tex2Dlod(diffuse_tex, UV1) * mips[0].y;
	result			+= tex2Dlod(diffuse_tex, UV2) * mips[1].y;
	result			+= tex2Dlod(diffuse_tex, UV3) * mips[2].y;
	result			+= tex2Dlod(diffuse_tex, UV4) * mips[3].y;
	
	// Chromatic aberration
	result.r = pow( result.r, 1.25 );
	result.g = pow( result.g, 1.10 );
	result.b = result.b;

	return result;
}

//_________________________________________________________________________________________________

float4 test_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float2 uv = In.uvSet01.xy - float2(0.5,0.5) - float2(0.5/256.0,0.5/256.0);
	float4 col = float4(0,0,0,0);
	if (length(uv) < (0.8/256.0))
	{
		col = float4(1,1,1,1);
	}
	return col;
}
//_________________________________________________________________________________________________

struct VS_OUTTHRESHOLD
{
	float4 position  : POSITION;	// Projected space position 
	float4 Texture  : TEXCOORD0;
};

struct PS_INTHRESHOLD
{
	float4 Texture : TEXCOORD0;
};

VS_OUTTHRESHOLD copytexthreshold_vx( VS_IN_2D input )
{
	VS_OUTTHRESHOLD Out;

	Out.position = input.position;
#ifdef _WINPC
	Out.Texture.xy = input.uvSet0;
#else
	Out.Texture.xy = input.position.xy*0.5 + 0.5;
	Out.Texture.y = 1.0 - Out.Texture.y;
#endif
	Out.Texture.z = threshold * threshold;
	Out.Texture.w = threshold;

	return Out;									// Transfer color
}

//_________________________________________________________________________________________________

float4 copytexthreshold_px( PS_INTHRESHOLD In ) : COLOR
{
	float4 color = tex2D(diffuse_tex, In.Texture.xy);
//	float luminance = (color.r + color.g + color.b) * 0.33334;
	float luminance = dot(color.rgb, float3(0.30, 0.59, 0.11));

	float lum = saturate((luminance - In.Texture.z) / (In.Texture.w - In.Texture.z));
//	float lum = smoothstep(Out.Texture.z, Out.Texture.w, luminance);
	return color * float4(lum, lum, lum, 1.0);
}

//_________________________________________________________________________________________________

float4 showDepthR32F_px(in VS_OUT_POS_UVSET01 In) : COLOR
{
	float4 depth = (tex2D(diffuse_tex, In.uvSet01.xy) - depth_modifiers.x) / (1.0 - depth_modifiers.x);
	return float4(depth.rgb, 1);
}

float4 showDepthD24_px(in VS_OUT_POS_UVSET01 In) : COLOR
{
	float4 depthColor = tex2D(diffuse_tex, In.uvSet01.xy).rgba;
	float3 depthFactor = float3(256.0*256.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0);
	float depth = dot(round(depthColor.rgb*255.0), depthFactor);

	float4 output = (depth - depth_modifiers.x) / (1.0 - depth_modifiers.x);
	return float4(output.rgb, 1);
}
